<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Courier;
use App\Models\Delivery;
use App\Models\Restaurant;
use App\Models\Zone;
use Carbon\Carbon;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $today = Carbon::today();
        $fromInput = $request->query('from');
        $toInput = $request->query('to');

        $rangeStart = $fromInput ? Carbon::parse($fromInput)->startOfDay() : $today->copy()->subDays(6)->startOfDay();
        $rangeEnd = $toInput ? Carbon::parse($toInput)->endOfDay() : $today->copy()->endOfDay();

        if ($rangeEnd->lessThan($rangeStart)) {
            [$rangeStart, $rangeEnd] = [$rangeEnd->copy()->startOfDay(), $rangeStart->copy()->endOfDay()];
        }

        $deliveriesRange = Delivery::query()->whereBetween('created_at', [$rangeStart, $rangeEnd]);

        $orderTotals = [
            'total' => (clone $deliveriesRange)->count(),
            'today' => Delivery::whereDate('created_at', $today)->count(),
            'pending' => (clone $deliveriesRange)->where('status', Delivery::STATUS_PENDING)->count(),
            'confirmed' => (clone $deliveriesRange)->where('status', Delivery::STATUS_CONFIRMED)->count(),
            'at_restaurant' => (clone $deliveriesRange)->where('status', Delivery::STATUS_AT_RESTAURANT)->count(),
            'en_route' => (clone $deliveriesRange)->where('status', Delivery::STATUS_EN_ROUTE)->count(),
            'delivered' => (clone $deliveriesRange)->where('status', Delivery::STATUS_DELIVERED)->count(),
            'cancelled' => (clone $deliveriesRange)->where('status', Delivery::STATUS_CANCELLED)->count(),
            'urgent' => (clone $deliveriesRange)->where('is_urgent', true)->count(),
        ];

        $ordersByDay = [];
        $cursor = $rangeStart->copy();
        while ($cursor->lte($rangeEnd)) {
            $date = $cursor->copy();
            $ordersByDay[] = [
                'date' => $date->format('Y-m-d'),
                'label' => $date->locale('es')->translatedFormat('d M'),
                'count' => Delivery::whereDate('created_at', $date)->count(),
            ];
            $cursor->addDay();
        }

        $topRestaurants = (clone $deliveriesRange)
            ->selectRaw('restaurant_id, COUNT(*) as total')
            ->whereNotNull('restaurant_id')
            ->groupBy('restaurant_id')
            ->orderByDesc('total')
            ->with('restaurant')
            ->limit(5)
            ->get();

        $completedRange = (clone $deliveriesRange)->where('status', Delivery::STATUS_DELIVERED);
        $incomeTotal = (clone $completedRange)->sum('delivery_fee');
        $incomeCash = (clone $completedRange)->where('payment_method', 'cash')->sum('delivery_fee');
        $incomeWallet = (clone $completedRange)->where('payment_method', 'wallet')->sum('delivery_fee');

        return view('admin.dashboard', [
            'zoneCount' => Zone::count(),
            'restaurantCount' => Restaurant::count(),
            'courierCount' => Courier::count(),
            'courierOnlineCount' => Courier::where('is_online', true)->count(),
            'orderTotals' => $orderTotals,
            'ordersByDay' => $ordersByDay,
            'topRestaurants' => $topRestaurants,
            'incomeTotal' => $incomeTotal,
            'incomeCash' => $incomeCash,
            'incomeWallet' => $incomeWallet,
            'rangeStart' => $rangeStart,
            'rangeEnd' => $rangeEnd,
            'statusCounts' => [
                'Pendientes' => $orderTotals['pending'] ?? 0,
                'Confirmadas' => $orderTotals['confirmed'] ?? 0,
                'En restaurante' => $orderTotals['at_restaurant'] ?? 0,
                'En camino' => $orderTotals['en_route'] ?? 0,
                'Entregadas' => $orderTotals['delivered'] ?? 0,
                'Canceladas' => $orderTotals['cancelled'] ?? 0,
            ],
            'paymentSplit' => [
                'Efectivo' => $incomeCash ?? 0,
                'Wallet' => $incomeWallet ?? 0,
            ],
        ]);
    }
}
