<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Delivery;
use App\Models\Courier;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function index(Request $request)
    {
        $query = Delivery::query()
            ->with(['restaurant.zone', 'courier.zone'])
            ->latest();

        $deliveries = $query->paginate(50);

        $latestDelivery = Delivery::query()
            ->with('restaurant')
            ->latest()
            ->first();

        $statsQuery = Delivery::query();
        $stats = [
            'total' => (int) $statsQuery->count(),
            'pending' => (int) (clone $statsQuery)->where('status', Delivery::STATUS_PENDING)->count(),
            'confirmed' => (int) (clone $statsQuery)->where('status', Delivery::STATUS_CONFIRMED)->count(),
            'at_restaurant' => (int) (clone $statsQuery)->where('status', Delivery::STATUS_AT_RESTAURANT)->count(),
            'en_route' => (int) (clone $statsQuery)->where('status', Delivery::STATUS_EN_ROUTE)->count(),
            'delivered' => (int) (clone $statsQuery)->where('status', Delivery::STATUS_DELIVERED)->count(),
            'urgent' => (int) (clone $statsQuery)->where('is_urgent', true)->count(),
        ];

        return view('admin.orders.index', [
            'deliveries' => $deliveries,
            'stats' => $stats,
            'latestDelivery' => $latestDelivery,
        ]);
    }

    public function poll(Request $request)
    {
        // Lightweight endpoint for admin alerts (sound/notifications).
        $since = now()->subDays(2);
        $latest = Delivery::query()
            ->where('created_at', '>=', $since)
            ->orderByDesc('id')
            ->first(['id', 'status', 'created_at', 'restaurant_id', 'customer_name', 'is_urgent']);

        $pendingCount = (int) Delivery::query()
            ->where('status', Delivery::STATUS_PENDING)
            ->count();

        return response()->json([
            'latest_id' => $latest?->id,
            'latest_status' => $latest?->status,
            'latest_created_at' => optional($latest?->created_at)->toIso8601String(),
            'latest_is_urgent' => (bool) ($latest?->is_urgent ?? false),
            'pending_count' => $pendingCount,
        ]);
    }

    public function show(Delivery $delivery)
    {
        $delivery->load(['restaurant.zone', 'courier.zone']);

        $couriers = Courier::query()
            ->with('zone')
            ->where('is_active', true)
            ->orderByDesc('is_online')
            ->orderBy('name')
            ->get();

        return view('admin.orders.show', [
            'delivery' => $delivery,
            'couriers' => $couriers,
        ]);
    }

    public function updateStatus(Request $request, Delivery $delivery)
    {
        $data = $request->validate([
            'status' => 'required|in:pending,confirmed,at_restaurant,en_route,delivered,cancelled',
            'cancel_reason' => 'nullable|string|max:1000',
        ]);

        $status = $data['status'];
        $update = [
            'status' => $status,
        ];

        if ($status === Delivery::STATUS_DELIVERED) {
            $update['completed_at'] = $delivery->completed_at ?? now();
            $update['cancelled_at'] = null;
            $update['cancel_reason'] = null;
        } elseif ($status === Delivery::STATUS_CANCELLED) {
            $update['cancelled_at'] = $delivery->cancelled_at ?? now();
            $update['cancel_reason'] = $data['cancel_reason'] ?: 'Cancelado por administrador';
            $update['completed_at'] = null;
        } else {
            $update['completed_at'] = null;
            $update['cancelled_at'] = null;
            $update['cancel_reason'] = null;
        }

        $delivery->update($update);

        return redirect()
            ->back()
            ->with('success', 'Estado de la orden actualizado.');
    }

    public function assignCourier(Request $request, Delivery $delivery)
    {
        $data = $request->validate([
            'courier_id' => 'required|exists:couriers,id',
        ]);

        $courier = Courier::findOrFail($data['courier_id']);

        $delivery->update([
            'courier_id' => $courier->id,
        ]);

        return redirect()
            ->back()
            ->with('success', 'Repartidor asignado correctamente.');
    }
}
